import os.path
import cmagic

class MagicException(Exception): pass

class Magic:
    """
    Magic is a wrapper around the libmagic C library.  
    
    """

    def __init__(self, mime=False, magic_file=None):
        """
        Create a new libmagic wrapper.

        mime - if True, mimetypes are returned instead of textual descriptions
        magic_file - use a mime database other than the system default
        
        """
        flags = cmagic.MAGIC_NONE
        if mime:
            flags |= cmagic.MAGIC_MIME
            
        self.cookie = cmagic.magic_open(flags)

        if magic_file:
            res = cmagic.magic_load(self.cookie, magic_file)
        else:
            res = cmagic.magic_load(self.cookie)

        if res == -1:
            self._check_error()


    def from_buffer(self, buf):
        """
        Identify the contents of `buf`
        """

        
        res = cmagic.magic_buffer(self.cookie, buf)

        if not res:
            self._check_error()
        else:
            return res

    def from_file(self, filename):
        """
        Identify the contents of file `filename`
        raises IOError if the file does not exist
        """

        res = cmagic.magic_file(self.cookie, filename)

        if not res:
            if not os.path.exists(filename):
                raise IOError(cmagic.magic_error(self.cookie))
            else:
                self._check_error()
        else:
            return res



    def __del__(self):
        cmagic.magic_close(self.cookie)

    def _check_error(self):
        err = cmagic.magic_error(self.cookie)
        if err:
            raise MagicException(err)


_magic_mime = None
_magic = None

def _get_magic_mime():
    global _magic_mime
    if not _magic_mime:
        _magic_mime = Magic(mime=True)
    return _magic_mime

def _get_magic():
    global _magic
    if not _magic:
        _magic = Magic()
    return _magic

def _get_magic_type(mime):
    if mime:
        return _get_magic_mime()
    else:
        return _get_magic()

def from_file(filename, mime=False):
    m = _get_magic_type(mime)
    return m.from_file(filename)

def from_buffer(buffer, mime=False):
    m = _get_magic_type(mime)
    return m.from_buffer(buffer)

